unit Sockets;
//***********************************************************************
//**  Sockets Component                                                **
//***********************************************************************
//**  This component provides a quick and easy way to begin            **
//**  implementing sockets in your applications.  The component was    **
//**  designed as a minimalist framework upon which more complex and   **
//**  customized asynchronous sockets could be constructed.  As a      **
//**  direct result of interest in this component, I have released it  **
//**  with a few quick code changes.  Therefore, although I believe it **
//**  appears fine, it may indeed be succeptible to a few 'features'.  **
//**  This component also demonstrates an internalized messaging       **
//**  structure that will hopefully be deemed useful.                  **
//**  The code contained in this document is entirely free to use in   **
//**  both commercial and non-commercial applications.  If this        **
//**  component is utilized in any way, shape, or form, I would        **
//**  appreciate a notification via email indicating this, as well as  **
//**  any comments or suggestions you may have.  Bug reports are       **
//**  encouraged.                                                      **
//**  I can be reached at:                                             **
//**    elvis@sway.com                                                 **
//**                                Sincerely,                         **
//**                                  T. J. Sobotka                    **
//**                                  March 1997                       **
//***********************************************************************
//**  Properties:                                                      **
//**    IPAddress:  A string representation of the IP dot notation     **
//**           address.                                                **
//**    PortNumber:  Word sized value to designate port number.        **
//**  Methods and Procedures:                                          **
//**    procedure DoClose                                              **
//**           Closes the socket.                                      **
//**    procedure DoReceive(Buffer: Pointer; var ReceiveLen: LongInt)  **
//**           Receives data.                                          **
//**    procedure DoSend(Buffer: Pointer; var SendLen: LongInt)        **
//**           Sends data.                                             **
//**    procedure DoListen                                             **
//**           Listens on the port as specified by PortNumber.         **
//**    procedure DoConnect                                            **
//**           Connects to the IPAddress on PortNumber.                **
//**    procedure DoAccept(var AcceptSocket: TAsyncSocket)             **
//**           Will set up the given socket for further use.           **
//**                                                                   **
//**    *NOTE*:  It is encouraged that you customize this code to suit **
//**           your personal needs.  Items that might be useful to     **
//**           alter for optimization purposes are the FD_ events.     **
//**           Simply remove the ones you feel you do not need.        **
//**           Be CAREFUL if you are unfamiliar with sockets.  All     **
//**           properties offer their related members as public for    **
//**           advanced users who wish to access them directly.  For   **
//**           anyone who does not wish to use them, simply use the    **
//**           aforementioned properties and procedures.  You can      **
//**           also undefine the SOCKETEXCEPTION to prevent the        **
//**           component from generating exceptions for unallocated    **
//**           events.  SOCKETEXCEPTION provided for debug purposes.   **
//***********************************************************************
interface

{$DEFINE SOCKETEXCEPTION}
uses
  Windows, Classes, Messages, Winsock, Forms, SysUtils;

const
  WM_SOCKET                 = WM_USER + 0;
  WM_SOCKETERROR            = WM_USER + 1;
  WM_SOCKETCLOSE            = WM_USER + 2;
  WM_SOCKETREAD             = WM_USER + 3;
  WM_SOCKETCONNECT          = WM_USER + 4;
  WM_SOCKETACCEPT           = WM_USER + 5;
  WM_SOCKETWRITE            = WM_USER + 6;
  WM_SOCKETOOB              = WM_USER + 7;
  WM_SOCKETLISTEN           = WM_USER + 8;
  WM_SOCKETTIMER            = WM_USER + 9;

  DEFAULT_BUFFER_SIZE       = 8192;

type

  ESocket = class(Exception);

  TWMSocket = record
    Msg: Word;
    case Integer of
    0: (
      SocketNumber: Word;
      SocketDataSize: LongInt;
      Result: Longint);
    1: (
      WParamLo: Byte;
      WParamHi: Byte;
      SocketEvent: Word;
      SocketError: Word;
      ResultLo: Word;
      ResultHi: Word);
  end;

  TSocketMessageEvent = procedure (SocketMessage: TWMSocket) of object;

  TSocketMemoryStream = class(TMemoryStream)
  public
    property Capacity;
  end;

  TSockStatus = (skIdle, skConnecting, skConnected, skListening);

  TAsyncSocket = class(TComponent)
  public
    {m_Buffer:       TSocketMemoryStream;}
    m_Handle:       TSocket;
    m_HWnd:         HWnd;
    m_SockAddr:     TSockAddr;
    m_SockStatus:   TSockStatus;

    FOnError:       TSocketMessageEvent;
    FOnAccept:      TSocketMessageEvent;
    FOnClose:       TSocketMessageEvent;
    FOnConnect:     TSocketMessageEvent;
    FOnRead:        TSocketMessageEvent;
    FOnWrite:       TSocketMessageEvent;
    FOnListen:      TSocketMessageEvent;
    FOnOOB:         TSocketMessageEvent;

    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;

    function GetPortNumber: Word;
    function GetIPAddress: String;
    function ErrorTest(Evaluation: LongInt): LongInt;
    function ErrToStr(Err: LongInt): String;
    
    procedure AllocateSocket;
    procedure Initialize;
    procedure DeInitialize;
    procedure SetPortNumber(NewPortNumber: Word);
    procedure SetIPAddress(NewIPAddress: String);

    procedure DoClose;
    procedure DoReceive(Buffer: Pointer; var ReceiveLen: LongInt);
    procedure DoSend(Buffer: Pointer; var SendLen: LongInt);
    procedure DoListen;
    procedure DoConnect;
    procedure DoAccept(var AcceptSocket: TAsyncSocket);

    // Message Handlers
    procedure HWndProcedure(var Message: TMessage);
    procedure Message_Error(var Message: TWMSocket); message WM_SOCKETERROR;
    procedure Message_Close(var Message: TWMSocket); message WM_SOCKETCLOSE;
    procedure Message_Accept(var Message: TWMSocket); message WM_SOCKETACCEPT;
    procedure Message_Read(var Message: TWMSocket); message WM_SOCKETREAD;
    procedure Message_Connect(var Message: TWMSocket); message WM_SOCKETCONNECT;
    procedure Message_Write(var Message: TWMSocket); message WM_SOCKETWRITE;
    procedure Message_OOB(var Message: TWMSocket); message WM_SOCKETOOB;
    procedure Message_Listen(var Message: TWMSocket); message WM_SOCKETLISTEN;
    procedure Message_Timer(var Message: TWMSocket); message WM_SOCKETTIMER;
  published
    property IPAddress: String read GetIPAddress write SetIPAddress;
    property PortNumber: Word read GetPortNumber write SetPortNumber;

    property OnError: TSocketMessageEvent read FOnError write FOnError;
    property OnAccept: TSocketMessageEvent read FOnAccept write FOnAccept;
    property OnClose: TSocketMessageEvent read FOnClose write FOnClose;
    property OnConnect: TSocketMessageEvent read FOnConnect write FOnConnect;
    property OnRead: TSocketMessageEvent read FOnRead write FOnRead;
    property OnWrite: TSocketMessageEvent read FOnWrite write FOnWrite;
    property OnOOB: TSocketMessageEvent read FOnOOB write FOnOOB;
    property OnListen: TSocketMessageEvent read FOnListen write FOnListen;
  end;

  procedure Register;

var
  InstanceCount: LongInt = 0;

implementation

constructor TAsyncSocket.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  m_SockStatus := skIdle;
  InstanceCount := InstanceCount + 1;
  Initialize;
end;  // constructor TAsyncSocket.Create

destructor TAsyncSocket.Destroy;
begin
  DeInitialize;
  InstanceCount := InstanceCount - 1;
  inherited Destroy;
end;  // destructor TAsyncSocket.Destroy;

function TAsyncSocket.GetIPAddress: String;
begin
  Result := INet_NToA(m_SockAddr.sin_addr);
end;  // function TAsyncSocket.GetIPAddress: String

function TAsyncSocket.GetPortNumber: Word;
begin
  Result := NToHS(m_SockAddr.sin_port);
end;  // function TAsyncSocket.GetPortNumber: Word

procedure TAsyncSocket.AllocateSocket;
begin
  if (m_Handle = INVALID_SOCKET) then
    begin
    m_Handle := ErrorTest(socket(AF_INET, SOCK_STREAM, 0));
    end;  // if (m_Handle = INVALID_SOCKET) then
end;  // procedure TAsyncSocket.AllocateSocket

function TAsyncSocket.ErrorTest(Evaluation: LongInt): LongInt;
var
  TempMessage: TWMSocket;
begin
  if ((Evaluation = SOCKET_ERROR) OR (Evaluation = INVALID_SOCKET)) then
    begin
    TempMessage.Msg := WM_SOCKETERROR;
    TempMessage.SocketError := WSAGetLastError;
    TempMessage.SocketNumber := m_Handle;
    Dispatch(TempMessage);
    Result := Evaluation;
    end  // if ((Evaluation = SOCKET_ERROR) OR (Evaluation = INVALID_SOCKET)) then
  else
    Result := Evaluation;
end;  // function ErrorTest(Evaluation: LongInt): LongInt;

procedure TAsyncSocket.Initialize;
var
  TempWSAData: TWSAData;
begin
  if (InstanceCount = 1) then ErrorTest(WSAStartup($101, TempWSAData));
  m_Handle := INVALID_SOCKET;
  m_SockAddr.sin_family := AF_INET;
  m_HWnd := AllocateHWnd(HWndProcedure);
end;  // procedure TAsyncSocket.Initialize

procedure TAsyncSocket.DeInitialize;
begin
  DoClose;
  if (InstanceCount = 1) then ErrorTest(WSACleanup);
  DeallocateHWnd(m_HWnd);
end;  // procedure TAsyncSocket.DeInitialize

procedure TAsyncSocket.SetIPAddress(NewIPAddress: String);
begin
  m_SockAddr.sin_addr := TInAddr(INet_Addr(PChar(NewIPAddress)));
end;  // procedure TAsyncSocket.SetIPAddress(NewIPAddress: String)

procedure TAsyncSocket.SetPortNumber(NewPortNumber: Word);
begin
  m_SockAddr.sin_port := HToNS(NewPortNumber);
end;  // procedure TAsyncSocket.SetPortNumber(NewPortNumber: Word)

procedure TAsyncSocket.DoReceive(Buffer: Pointer; var ReceiveLen: LongInt);
begin
  ReceiveLen := recv(m_Handle, Buffer^, ReceiveLen, 0);
  ErrorTest(ReceiveLen);
end;  // TAsyncSocket.DoReceive(Buffer: Pointer; BufferLen: LongInt)

procedure TAsyncSocket.DoSend(Buffer: Pointer; var SendLen: LongInt);
begin
  SendLen := send(m_Handle, Buffer^, SendLen, 0);
  ErrorTest(SendLen);
end;  // procedure TAsyncSocket.DoSend(Buffer: Pointer; BufferLen: LongInt)

procedure TAsyncSocket.DoClose;
var
  TempMessage: TWMSocket;
begin
  if (m_Handle <> INVALID_SOCKET) then
    begin
    TempMessage.Msg := WM_SOCKETCLOSE;
    TempMessage.SocketNumber := m_Handle;
    ErrorTest(closesocket(m_Handle));
    m_Handle := INVALID_SOCKET;
    m_SockStatus := skIdle;
    Dispatch(TempMessage);
    end;  // if (m_Handle <> INVALID_SOCKET) then
end;  // procedure TAsyncSocket.DoClose

procedure TAsyncSocket.DoAccept(var AcceptSocket: TAsyncSocket);
var
  TempSize: Integer;
begin
  TempSize := SizeOf(TSockAddr);
  AcceptSocket.m_Handle := accept(m_Handle, AcceptSocket.m_SockAddr,
    TempSize);
  AcceptSocket.ErrorTest(WSAAsyncSelect(AcceptSocket.m_Handle,
    AcceptSocket.m_HWnd, WM_SOCKET,
    FD_READ OR FD_CLOSE OR FD_CONNECT OR FD_OOB OR FD_WRITE));
end;  // procedure TAsyncSocket.DoAccept(var AcceptSocket: TAsyncSocket)

procedure TAsyncSocket.DoListen;
var
  TempMessage: TWMSocket;
begin
  DoClose;
  AllocateSocket;
  ErrorTest(WSAAsyncSelect(m_Handle, m_HWnd, WM_SOCKET, FD_ACCEPT));
  ErrorTest(bind(m_Handle, m_SockAddr, SizeOf(TSockAddr)));
  ErrorTest(listen(m_Handle, 5));
  m_SockStatus := skListening;
  TempMessage.Msg := WM_SOCKETLISTEN;
  TempMessage.SocketNumber := m_Handle;
  Dispatch(TempMessage);
end;  // procedure TAsyncSocket.DoListen(PortNumber: Word)

procedure TAsyncSocket.DoConnect;
var
  TempResult: LongInt;
begin
  DoClose;
  AllocateSocket;
  ErrorTest(WSAAsyncSelect(m_Handle, m_HWnd, WM_SOCKET,
    FD_READ OR FD_CLOSE OR FD_CONNECT OR FD_OOB OR FD_WRITE));
  TempResult := connect(m_Handle, m_SockAddr, SizeOf(TSockAddr));
  m_SockStatus := skConnecting;
  if ((TempResult = SOCKET_ERROR) AND (WSAGetLastError <> WSAEWOULDBLOCK)) then
    ErrorTest(SOCKET_ERROR);
end;  // procedure TAsyncSocket.DoConnect(IPAddress: String; PortNumber: Word)

procedure TAsyncSocket.HWndProcedure(var Message: TMessage);
var
  TempMessage: TWMSocket;
begin
  case Message.Msg of
    WM_SOCKET:
      begin
      if (Message.LParamHi > WSABASEERR) then
        begin
        WSASetLastError(Message.LParamHi);
        ErrorTest(SOCKET_ERROR);
        end  // if (Message.LParamHi > WSABASEERR) then
      else
        begin
        case Message.LParamLo of
          FD_READ:
            begin
            TempMessage.SocketDataSize := 0;
            ErrorTest(IOCtlSocket(m_Handle, FIONREAD, TempMessage.SocketDataSize));
            TempMessage.Msg := WM_SOCKETREAD;
            TempMessage.SocketNumber := m_Handle;
            Dispatch(TempMessage);
            end;  // FD_READ
          FD_CLOSE:
            begin
            TempMessage.Msg := WM_SOCKETCLOSE;
            TempMessage.SocketNumber := m_Handle;
            Dispatch(TempMessage);
            end; // FD_CLOSE
          FD_CONNECT:
            begin
            TempMessage.Msg := WM_SOCKETCONNECT;
            TempMessage.SocketNumber := m_Handle;
            m_SockStatus := skConnected;
            Dispatch(TempMessage);
            end;  // FD_CONNECT
          FD_ACCEPT:
            begin
            TempMessage.Msg := WM_SOCKETACCEPT;
            TempMessage.SocketNumber := m_Handle;
            Dispatch(TempMessage);
            end;  // FD_ACCEPT
          FD_WRITE:
            begin
            TempMessage.Msg := WM_SOCKETWRITE;
            TempMessage.SocketNumber := m_Handle;
            Dispatch(TempMessage);
            end;  // FD_WRITE
          FD_OOB:
            begin
            TempMessage.Msg := WM_SOCKETOOB;
            TempMessage.SocketNumber := m_Handle;
            Dispatch(TempMessage);
            end;  // FD_OOB
          end;  // case Message.LParamLo of
        end  // else (if (Message.LParamHi > WSABASEERR) then)
      end;  // WM_SOCKET:
    end;  // case Message.Msg of
end;  // procedure TAsyncSocket.HWndProcedure(var Message: TMessage)

procedure TAsyncSocket.Message_Error(var Message: TWMSocket);
begin
  if Assigned(FOnError) then FOnError(Message)
  {$IFNDEF SOCKETEXCEPTION}
  ;
  {$ELSE}
  else
  Application.ShowException(
    ESocket.Create(ErrToStr(Message.SocketError) + ' on socket ' +
    IntToStr(Message.SocketNumber)));
  {$ENDIF}
end;  // procedure TAsyncSocket.Message_Error(var Message: TMessage)

procedure TAsyncSocket.Message_Close(var Message: TWMSocket);
begin
  if Assigned(FOnClose) then FOnClose(Message)
  {$IFNDEF SOCKETEXCEPTION}
  ;
  {$ELSE}
  else
  Application.ShowException(
    ESocket.Create('WM_SOCKETCLOSE on socket ' + IntToStr(Message.SocketNumber)));
  {$ENDIF}
end;  // procedure TAsyncSocket.Message_Close(var Message: TWMSocket)

procedure TAsyncSocket.Message_Accept(var Message: TWMSocket);
begin
  if Assigned(FOnAccept) then FOnAccept(Message)
  {$IFNDEF SOCKETEXCEPTION}
  ;
  {$ELSE}
  else
  Application.ShowException(
    ESocket.Create('WM_SOCKETACCEPT on socket ' + IntToStr(Message.SocketNumber)));
  {$ENDIF}
end;  // procedure TAsyncSocket.Message_Accept(var Message: TWMSocket)

procedure TAsyncSocket.Message_Read(var Message: TWMSocket);
begin
  if Assigned(FOnRead) then FOnRead(Message) 
  {$IFNDEF SOCKETEXCEPTION}
  ;
  {$ELSE}
  else
  Application.ShowException(
    ESocket.Create('WM_SOCKETREAD on socket ' + IntToStr(Message.SocketNumber)));
  {$ENDIF}
end;  // procedure TAsyncSocket.Message_Read(var Message: TWMSocket)

procedure TAsyncSocket.Message_Connect(var Message: TWMSocket);
begin 
  if Assigned(FOnConnect) then FOnConnect(Message) 
  {$IFNDEF SOCKETEXCEPTION}
  ;
  {$ELSE}
  else
  Application.ShowException(
    ESocket.Create('WM_SOCKETCONNECT on socket ' + IntToStr(Message.SocketNumber)));
  {$ENDIF}
end;  // procedure TAsyncSocket.Message_Connect(var Message: TWMSocket)

procedure TAsyncSocket.Message_Write(var Message: TWMSocket);
begin              
  if Assigned(FOnWrite) then FOnWrite(Message) 
  {$IFNDEF SOCKETEXCEPTION}
  ;
  {$ELSE}
  else
  Application.ShowException(
    ESocket.Create('WM_SOCKETWRITE on socket ' + IntToStr(Message.SocketNumber)));
  {$ENDIF}
end;  // procedure TAsyncSocket.Message_Write(var Message: TWMSocket)

procedure TAsyncSocket.Message_OOB(var Message: TWMSocket);
begin
  if Assigned(FOnOOB) then FOnOOB(Message) 
  {$IFNDEF SOCKETEXCEPTION}
  ;
  {$ELSE}
  else
  Application.ShowException(
    ESocket.Create('WM_SOCKETOOB on socket ' + IntToStr(Message.SocketNumber)));
  {$ENDIF}
end;  // procedure TAsyncSocket.Message_OOB(var Message: TWMSocket)

procedure TAsyncSocket.Message_Listen(var Message: TWMSocket);
begin        
  if Assigned(FOnListen) then FOnListen(Message) 
  {$IFNDEF SOCKETEXCEPTION}
  ;
  {$ELSE}
  else
  Application.ShowException(
    ESocket.Create('WM_SOCKETLISTEN on socket ' + IntToStr(Message.SocketNumber)));
  {$ENDIF}
end;  // procedure TAsyncSocket.Message_Listen(var Message: TWMSocket)

procedure TAsyncSocket.Message_Timer(var Message: TWMSocket);
begin
  Application.ShowException(
    ESocket.Create('WM_SOCKETTIMER on socket ' + IntToStr(Message.SocketNumber)));
end;  // procedure TAsyncSocket.Message_Timer(var Message: TWMSocket)

function TAsyncSocket.ErrToStr(Err: LongInt): String;
begin
  case Err of
    WSAEINTR:
      Result := 'WSAEINTR';
    WSAEBADF:
      Result := 'WSAEBADF';
    WSAEACCES:
      Result := 'WSAEACCES';
    WSAEFAULT:
      Result := 'WSAEFAULT';
    WSAEINVAL:
      Result := 'WSAEINVAL';
    WSAEMFILE:
      Result := 'WSAEMFILE';
    WSAEWOULDBLOCK:
      Result := 'WSAEWOULDBLOCK';
    WSAEINPROGRESS:
      Result := 'WSAEINPROGRESS';
    WSAEALREADY:
      Result := 'WSAEALREADY';
    WSAENOTSOCK:
      Result := 'WSAENOTSOCK';
    WSAEDESTADDRREQ:
      Result := 'WSAEDESTADDRREQ';
    WSAEMSGSIZE:
      Result := 'WSAEMSGSIZE';
    WSAEPROTOTYPE:
      Result := 'WSAEPROTOTYPE';
    WSAENOPROTOOPT:
      Result := 'WSAENOPROTOOPT';
    WSAEPROTONOSUPPORT:
      Result := 'WSAEPROTONOSUPPORT';
    WSAESOCKTNOSUPPORT:
      Result := 'WSAESOCKTNOSUPPORT';
    WSAEOPNOTSUPP:
      Result := 'WSAEOPNOTSUPP';
    WSAEPFNOSUPPORT:
      Result := 'WSAEPFNOSUPPORT';
    WSAEAFNOSUPPORT:
      Result := 'WSAEAFNOSUPPORT';
    WSAEADDRINUSE:
      Result := 'WSAEADDRINUSE';
    WSAEADDRNOTAVAIL:
      Result := 'WSAEADDRNOTAVAIL';
    WSAENETDOWN:
      Result := 'WSAENETDOWN';
    WSAENETUNREACH:
      Result := 'WSAENETUNREACH';
    WSAENETRESET:
      Result := 'WSAENETRESET';
    WSAECONNABORTED:
      Result := 'WSAECONNABORTED';
    WSAECONNRESET:
      Result := 'WSAECONNRESET';
    WSAENOBUFS:
      Result := 'WSAENOBUFS';
    WSAEISCONN:
      Result := 'WSAEISCONN';
    WSAENOTCONN:
      Result := 'WSAENOTCONN';
    WSAESHUTDOWN:
      Result := 'WSAESHUTDOWN';
    WSAETOOMANYREFS:
      Result := 'WSAETOOMANYREFS';
    WSAETIMEDOUT:
      Result := 'WSAETIMEDOUT';
    WSAECONNREFUSED:
      Result := 'WSAECONNREFUSED';
    WSAELOOP:
      Result := 'WSAELOOP';
    WSAENAMETOOLONG:
      Result := 'WSAENAMETOOLONG';
    WSAEHOSTDOWN:
      Result := 'WSAEHOSTDOWN';
    WSAEHOSTUNREACH:
      Result := 'WSAEHOSTUNREACH';
    WSAENOTEMPTY:
      Result := 'WSAENOTEMPTY';
    WSAEPROCLIM:
      Result := 'WSAEPROCLIM';
    WSAEUSERS:
      Result := 'WSAEUSERS';
    WSAEDQUOT:
      Result := 'WSAEDQUOT';
    WSAESTALE:
      Result := 'WSAESTALE';
    WSAEREMOTE:
      Result := 'WSAEREMOTE';
    WSASYSNOTREADY:
      Result := 'WSASYSNOTREADY';
    WSAVERNOTSUPPORTED:
      Result := 'WSAVERNOTSUPPORTED';
    WSANOTINITIALISED:
      Result := 'WSANOTINITIALISED';
    WSAHOST_NOT_FOUND:
      Result := 'WSAHOST_NOT_FOUND';
    WSATRY_AGAIN:
      Result := 'WSATRY_AGAIN';
    WSANO_RECOVERY:
      Result := 'WSANO_RECOVERY';
    WSANO_DATA:
      Result := 'WSANO_DATA';
    else Result := 'UNDEFINED WINSOCK ERROR';
    end;  // case Err of
end;  // function TAsyncSocket.ErrToStr(Err: LongInt): String

procedure Register;
begin
  RegisterComponents('Sockets', [TAsyncSocket]);
end;

end.
 